﻿/*
** Copyright (C) 2011 - 2015 PROMESS
** Gesellschaft für Montage und Prüfsysteme mbH
** ALL RIGHTS RESERVED
** Developers:
** Andreas Groch
*/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;

namespace AstasDllCSharpTest.Astas
{
    internal static class AstasNativeMethods
    {
        // Das "EntryPoint" Attribut wurde benutzt um die originalen DLL Funktionen umzubenennen
        // damit die public Funktionen der Klasse die gleichen Namen wie die DLL Funktionen erhalten 
        // konnten

        // http://msdn.microsoft.com/en-us/library/f5xe74x8%28v=vs.71%29.aspx
        // An entry point identifies the location of a function in a DLL. Within a managed project,
        // the original name or ordinal entry point of a target function identifies that function
        // across the interoperation boundary. Further, you can map the entry point to a different name, effectively renaming the function.
        // Following is a list of possible reasons to rename a DLL function:
        // * To avoid using case-sensitive API function names
        // * To comply with existing naming standards
        // * To accommodate functions that take different data types (by declaring multiple versions of the same DLL function)
        // * To simplify using APIs that contain ANSI and Unicode versions

        /// <summary>
        /// Initialisierung der ASTAS-DLL. Dieser Schritt ist notwendig für eine korrekte Funktion
        /// der ASTAS-DLL.
        /// Hinweis: Diese Funktion muss zwingend als erstes zur Initialisierung der DLL-
        /// Funktionen aufgerufen werden. Sie kann nur einmal ausgeführt werden.
        /// Rückgabe NoErr/Err
        /// </summary>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllInit")]
        internal static extern int Init();

        /// <summary>
        /// Re-Initialisierung der ASTAS_DLL.
        /// Rückgabe NoErr/Err
        /// </summary>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllReInit")]
        internal static extern int ReInit();

        /// <summary>
        /// Mit dieser Funktion werden alle angeschlossenen Geräte AE703 (i.A. max. 16) gesucht
        /// und in der DeviceInfo-Struktur angelegt. Diese Struktur muß vom aufrufenden Programm
        /// als Zeiger/Pointer übergeben werden.
        /// Rückgabe: Anzahl der gefundenen Geräte
        /// </summary>
        /// <param name="deviceInfo"></param>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllSearch", CallingConvention = CallingConvention.Cdecl)]
        internal static extern int Search(out DeviceInfo deviceInfo);

        /// <summary>
        /// Öffnen des gewählten Gerätes.
        /// Rückgabe NoErr/ErrDllNotInit
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllOpen", CallingConvention = CallingConvention.Cdecl)]
        internal static extern int Open(int list_idx);

        /// <summary>
        /// Schließen des gewählten Gerätes.
        /// Rückgabe NoErr/ErrDllNotInit/ErrDeviceNotOpen
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllClose", CallingConvention = CallingConvention.Cdecl)]
        internal static extern int Close(int list_idx);

        /// <summary>
        /// Lesen der aktuellen Messwerte des gewählten Gerätes. Es muß die Struktur DeviceData
        /// als Zeiger/Pointer übergeben werden.
        /// Rückgabe: NoErr/Err/ErrDllNotInit/ErrDeviceNotOpen
        /// </summary>
        /// <param name="listIdx"></param>
        /// <param name="deviceData"></param>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllReadData", CallingConvention = CallingConvention.Cdecl)]
        internal static extern int ReadData(int list_idx, out DeviceData deviceData);

        /// <summary>
        /// Nullung des gewählten Gerätes.
        /// Rückgabe: NoErr/ErrDllNotInit/ErrDeviceNotOpen
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllNull", CallingConvention = CallingConvention.Cdecl)]
        internal static extern int Null(int list_idx);

        /// <summary>
        /// Umschaltung zwischen Brutto/Netto des ausgewählten Gerätes.
        /// Rückgabe: NoErr/ErrDllNotInit/ErrDeviceNotOpen/ErrFuncNotImpl
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllGrossNet", CallingConvention = CallingConvention.Cdecl)]
        internal static extern int GrossNet(int list_idx);

        /// <summary>
        /// Umschaltung der Maßeinheit des ausgewählten Gerätes.
        /// Rückgabe: NoErr/ErrDllNotInit/ErrDeviceNotOpen
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllChangeUnit", CallingConvention = CallingConvention.Cdecl)]
        internal static extern int ChangeUnit(int list_idx);

        /// <summary>
        /// Umschaltung des Messbereichs des ausgewählten Gerätes.
        /// Rückgabe: NoErr/ErrDllNotInit/ErrDeviceNotOpen
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllChangeRange", CallingConvention = CallingConvention.Cdecl)]
        internal static extern int ChangeRange(int list_idx);

        /// <summary>
        /// Bestätigung der Statusbits ChangeUnit/ChangeRange. Gerät setzt darauf hin beide Statusbits zurück.
        /// Rückgabe: NO_ERR / Fehlercode
        /// </summary>
        /// <param name="list_idx"></param>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllAckChangeStatusBits", CallingConvention = CallingConvention.Cdecl)]
        internal static extern int AckChangeStatusBits(int list_idx);

        /// <summary>
        /// Reset des ausgewählten Gerätes. Danach nuß das Gerät neu gesucht und geöffnet werden.
        /// Rückgabe: NoErr
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllReset", CallingConvention = CallingConvention.Cdecl)]
        internal static extern int Reset(int list_idx);

        /// <summary>
        /// Abfage der ASTAS-DLL-Version.
        /// Rückgabe: DLL-Version
        /// </summary>
        /// <param></param>
        /// <returns></returns>
        [DllImport("astas_dll.dll", EntryPoint = "AstasDllDLLVersion", CallingConvention = CallingConvention.Cdecl)]
        internal static extern int DLLVersion();
    }
}
