﻿/*
** Copyright (C) 2011 - 2015 PROMESS
** Gesellschaft für Montage und Prüfsysteme mbH
** ALL RIGHTS RESERVED
** Developers:
** Andreas Groch
*/

using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;

namespace AstasDllCSharpTest.Astas
{
    // Der Text für die Beschreibung der Funktionen und Datenstrukturen
    // stammt aus der Bedienungsanleitung der ASTAS-DLL von A.S.T (www.ast.de).

    // DllImport Deklarationen, StructLayout und Marshal Attribute wurden mit 
    // http://clrinterop.codeplex.com/releases/view/14120 (PInvoke Interop Assistent)
    // erstellt.

    public static class AstasMethods
    {
        #region Public Methods

        /// <summary>
        /// Initialisierung der ASTAS-DLL. Dieser Schritt ist notwendig für eine korrekte Funktion
        /// der ASTAS-DLL.
        /// Hinweis: Diese Funktion muss zwingend als erstes zur Initialisierung der DLL-
        /// Funktionen aufgerufen werden. Sie kann nur einmal ausgeführt werden.
        /// Rückgabe NoErr/Err
        /// </summary>
        /// <returns></returns>
        public static ErrorCode Init()
        {
            return (ErrorCode)AstasNativeMethods.Init();
        }

        /// <summary>
        /// Re-Initialisierung der ASTAS_DLL.
        /// Rückgabe NoErr/Err
        /// </summary>
        /// <returns></returns>
        [SuppressMessage("Microsoft.Naming", "CA1709:IdentifiersShouldBeCasedCorrectly", MessageId = "Re")]
        public static ErrorCode ReInit()
        {
            return (ErrorCode)AstasNativeMethods.ReInit();
        }

        /// <summary>
        /// Mit dieser Funktion werden alle angeschlossenen Geräte AE703 (i.A. max. 16) gesucht
        /// und in der DeviceInfo-Struktur angelegt. Diese Struktur muß vom aufrufenden Programm
        /// als Zeiger/Pointer übergeben werden.
        /// Rückgabe: Anzahl der gefundenen Geräte
        /// </summary>
        /// <param name="deviceInfo"></param>
        /// <returns></returns>
        [SuppressMessage("Microsoft.Design", "CA1045:DoNotPassTypesByReference", MessageId = "0#")]
        public static int Search(out DeviceInfo deviceInfo)
        {
            return AstasNativeMethods.Search(out deviceInfo);
        }

        /// <summary>
        /// Öffnen des gewählten Gerätes.
        /// Rückgabe NoErr/ErrDllNotInit
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        public static ErrorCode Open(int listIdx)
        {
            return (ErrorCode)AstasNativeMethods.Open(listIdx);
        }

        /// <summary>
        /// Schließen des gewählten Gerätes.
        /// Rückgabe NoErr/ErrDllNotInit/ErrDeviceNotOpen
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        public static ErrorCode Close(int listIdx)
        {
            return (ErrorCode)AstasNativeMethods.Close(listIdx);
        }

        /// <summary>
        /// Lesen der aktuellen Messwerte des gewählten Gerätes. Es muß die Struktur DeviceData
        /// als Zeiger/Pointer übergeben werden.
        /// Rückgabe: NoErr/Err/ErrDllNotInit/ErrDeviceNotOpen
        /// </summary>
        /// <param name="listIdx"></param>
        /// <param name="deviceData"></param>
        /// <returns></returns>
        [SuppressMessage("Microsoft.Design", "CA1045:DoNotPassTypesByReference", MessageId = "1#")]
        public static ErrorCode ReadData(int listIdx, out DeviceData deviceData)
        {
            return (ErrorCode)AstasNativeMethods.ReadData(listIdx, out deviceData);
        }

        /// <summary>
        /// Nullung des gewählten Gerätes.
        /// Rückgabe: NoErr/ErrDllNotInit/ErrDeviceNotOpen
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        public static ErrorCode Null(int listIdx)
        {
            return (ErrorCode)AstasNativeMethods.Null(listIdx);
        }

        /// <summary>
        /// Umschaltung zwischen Brutto/Netto des ausgewählten Gerätes.
        /// Rückgabe: NoErr/ErrDllNotInit/ErrDeviceNotOpen/ErrFuncNotImpl
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        public static ErrorCode GrossNetto(int listIdx)
        {
            return (ErrorCode)AstasNativeMethods.GrossNet(listIdx);
        }
        
        /// <summary>
        /// Umschaltung der Maßeinheit des ausgewählten Gerätes.
        /// Rückgabe: NoErr/ErrDllNotInit/ErrDeviceNotOpen
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        public static ErrorCode ChangeUnit(int listIdx)
        {
            return (ErrorCode)AstasNativeMethods.ChangeUnit(listIdx);
        }

        /// <summary>
        /// Umschaltung des Messbereichs des ausgewählten Gerätes.
        /// Rückgabe: NoErr/ErrDllNotInit/ErrDeviceNotOpen
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        public static ErrorCode ChangeRange(int listIdx)
        {
            return (ErrorCode)AstasNativeMethods.ChangeRange(listIdx);
        }

        /// <summary>
        /// Reset des ausgewählten Gerätes. Danach nuß das Gerät neu gesucht und geöffnet werden.
        /// Rückgabe: NoErr
        /// </summary>
        /// <param name="listIdx"></param>
        /// <returns></returns>
        public static ErrorCode Reset(int listIdx)
        {
            return (ErrorCode)AstasNativeMethods.Reset(listIdx);
        }
        
        /// <summary>
        /// VErsion der ASTAS-DLL abfragen
        /// Rückgabe: -
        /// </summary>
        /// <param name="-"></param>
        /// <returns></returns>
        public static DllVersion DllVersion()
        {
            DllVersion dllVersion = new DllVersion(AstasNativeMethods.DLLVersion());
            
            return dllVersion;
        }

        #endregion Public Methods
    }
}
